//------------------------
// PARAMETERS

float4x4 gWorldViewProjMatrix
<
   string paramClass="intrinsic";
   string paramName="worldViewProj";
> = float4x4(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);

float4x4 gWorldViewMatrix
<
   string paramClass="intrinsic";
   string paramName="worldView";
> = float4x4(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);

float4x4 gWorldViewProjScreenMatrix
<
   string paramClass="intrinsic";
   string paramName="worldViewProjScreen";
> = float4x4(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);

float4 gModelSpaceCameraPos
<
   string paramClass="intrinsic";
   string paramName="modelSpaceCameraPos";
> = float4(0,0,0,1);

float4x4 gWorldShadowMatrix
<
   string paramClass="intrinsic";
   string paramName="shadowMatrix";
> = float4x4(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);

float4 gShadowXBiasYRange
<
   string paramClass="intrinsic";
   string paramName="shadow_XRange_YBias";
> = float4(0, 0, 0, 0);

float4x4 gLightProjectedMatrix
<
   string paramClass="intrinsic";
   string paramName="lightProjectedMatrix";
> = float4x4(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);

float4 gSunDir
<
   string paramClass="intrinsic";
   string paramName="sunDirection";
> = float4(0, 0, 1, 0);

float4 gFakeSunDir
<
   string paramClass="intrinsic";
   string paramName="fakeSunDirection";
> = float4(0, 0, 1, 0);

float4 gFakeSunParams
<
  string paramClass="intrinsic";
  string paramName="fakeSunParams";
> = float4(1, 0, 1, 0);

float gRGBScale
<
   string paramClass="intrinsic";
   string paramName="RGBScale";
   string paramType="float";
> = 1.0f;

float gFogDensity
<
   string paramClass="intrinsic";
   string paramName="FogDensity";
   string paramType="float";
> = 0.0f;

float gFogStart
<
   string paramClass="intrinsic";
   string paramName="FogStart";
   string paramType="float";
> = 0.0;

float gPlanarFogDensity
<
  string paramClass="intrinsic";
  string paramName="PlanarFogDensity";
  string paramType="float";
> = 0.0;

float4 gPlanarFogEquation
<
  string paramClass="intrinsic";
  string paramName="PlanarFogEquation";
> = float4(0,1,0,0);

float4 gWaterTexMtxParams
<
   string paramClass="param";
   string paramName="textureMatrixParams";
> = float4(1, 0, 0, 1);

float4 gWaterColorScales
<
   string paramClass="param";
   string paramName="colorScales";
> = float4(0, 0, 0, 0);

float4 gWaterBumpScales
<
   string paramClass="param";
   string paramName="bumpScales";
> = float4(0, 0, 0, 0);

float4 gWaterDeepColor
<
   string paramClass="param";
   string paramName="deepColor";
   string paramType="color";
   bool linearColor=true;
> = float4(0, 0, 0, 0);

float4 gWaterShallowColor
<
   string paramClass="param";
   string paramName="shallowColor";
   string paramType="color";
   bool linearColor=true;
> = float4(0, 0, 0, 0);

float4 gPlanarFogColor
<
string paramClass="intrinsic";
string paramName="PlanarFogColor";
string paramType="color";
> = float4(0,0,0,1);

float4 gFogColor
<
   string paramClass="intrinsic";
   string paramName="FogColor";
   string paramType="color";
> = float4(0,0,0,1);

float4 gShadowColor
<
   string paramClass="intrinsic";
   string paramName="shadowColor";
   string paramType="color";
> = float4(0,0,0,0);

float4 gShadowPCFOffset0
<
   string paramClass="intrinsic";
   string paramName="ShadowPCFOffset0";
> = float4(0.5,0.5,0.5,0.5);

float4 gShadowPCFOffset1
<
   string paramClass="intrinsic";
   string paramName="ShadowPCFOffset1";
> = float4(0,0,0,0);

float4 gShadowPCFOffset2
<
   string paramClass="intrinsic";
   string paramName="ShadowPCFOffset2";
> = float4(0,0,0,0);


sampler normalMap             : register(s0);
sampler skyCube               : register(s1);
sampler reflection            : register(s2);
sampler foam                  : register(s3);
sampler waveDistortionMap     : register(s4);
sampler gSampler7             : register(s7);

// END PARAMETERS


//------------------------
// SUBROUTINES

vec4 SRGBToLinear(vec4 c)
{
   vec4 ret = c;
   ret.rgb *= ret.rgb;
   return ret;
}

float ComputeSquaredExponentialFogDensity(float fogDistance)
{
   float  squaredExponent = fogDistance * fogDistance;      
   float  finalFogDensity = clamp( exp(-squaredExponent), 0.0, 1.0);
   return finalFogDensity;   
}

#ifdef SHADOWING_NV

vec3 getShadowValueNV( sampler2DShadow tx, vec4 texCoord0, vec4 texCoord1, vec4 texCoord2, float shadowColor )
{
	vec3 shadowVals;
	vec4 averageVec = vec4(0.25);
	
	vec4 shadowVec = shadow2DProj(tx, texCoord0);
	shadowVals.x = dot( shadowVec, averageVec );
	
	shadowVec = shadow2DProj(tx, texCoord1);
	shadowVals.y = dot( shadowVec, averageVec );
	
	shadowVec = shadow2DProj(tx, texCoord2);
	shadowVals.z = dot( shadowVec, averageVec );
	
	float shadowScalar = dot( shadowVals, vec3(0.33333) );

	shadowScalar = mix(shadowColor, 1.0, shadowScalar);
	return( vec3(shadowScalar) );
}

#else

#ifdef SHADOWMAP_SIZE
const vec2 shadow_scale = vec2( float(SHADOWMAP_SIZE), float(SHADOWMAP_SIZE) );
const vec2 shadow_inv_scale = vec2( float(INV_SHADOWMAP_SIZE), float(INV_SHADOWMAP_SIZE) );
#endif

vec3 getShadowingValue(sampler2D tx, vec4 shadowmapTexCoord, float lightSpacePixelDepth, float shadowColor )
{
	// Invert y-coord from shadow map for GL.
	shadowmapTexCoord.y = 1.0 - shadowmapTexCoord.y;

#if defined(_SHADOWS_FILTER_2x2_BILINEAR_)

	shadowmapTexCoord.xy *= shadow_scale;
	vec2 fracCoords = fract( shadowmapTexCoord.xy );
	shadowmapTexCoord.xy -= fracCoords;
	shadowmapTexCoord.xy *= shadow_inv_scale;

	vec4 shadowDepth;
	shadowDepth.x = texture2D( tx, shadowmapTexCoord.xy ).x;
	shadowDepth.y = texture2D( tx, vec2(shadowmapTexCoord.x, shadowmapTexCoord.y + SHADOWMAP_TEXOFFSET) ).x;
	shadowDepth.z = texture2D( tx, vec2(shadowmapTexCoord.x + SHADOWMAP_TEXOFFSET, shadowmapTexCoord.y) ).x;
	shadowDepth.w = texture2D( tx, vec2(shadowmapTexCoord.x + SHADOWMAP_TEXOFFSET, shadowmapTexCoord.y + SHADOWMAP_TEXOFFSET) ).x;

	vec4 shadows = step( lightSpacePixelDepth, shadowDepth  );

	vec2 temp = mix(shadows.xy, shadows.zw, fracCoords.x);
	float sv = mix(temp.x, temp.y, fracCoords.y);

#elif defined(_SHADOWS_FILTER_3x3_BILINEAR_)
	
	shadowmapTexCoord.xy *= shadow_scale;
	vec2 fracCoords = fract( shadowmapTexCoord.xy );
	shadowmapTexCoord.xy -= fracCoords;
	shadowmapTexCoord.xy *= shadow_inv_scale;

	vec3 leftCol, midCol, rightCol;
	//-- Sampling
	// Column 1
	leftCol.x = texture2D(tx, vec2( shadowmapTexCoord.x - SHADOWMAP_TEXOFFSET,
									shadowmapTexCoord.y - SHADOWMAP_TEXOFFSET)   ).x;
	leftCol.y = texture2D(tx, vec2( shadowmapTexCoord.x - SHADOWMAP_TEXOFFSET,
									shadowmapTexCoord.y)   ).x;
	leftCol.z = texture2D(tx, vec2( shadowmapTexCoord.x - SHADOWMAP_TEXOFFSET,
									shadowmapTexCoord.y + SHADOWMAP_TEXOFFSET)   ).x;
	// Column 2
	midCol.x = texture2D(tx, vec2( shadowmapTexCoord.x,
									shadowmapTexCoord.y - SHADOWMAP_TEXOFFSET)   ).x;
	midCol.y = texture2D(tx, vec2( shadowmapTexCoord.x,
									shadowmapTexCoord.y)   ).x;
	midCol.z = texture2D(tx, vec2( shadowmapTexCoord.x,
									shadowmapTexCoord.y + SHADOWMAP_TEXOFFSET)   ).x;
	// Column 3
	rightCol.x = texture2D(tx, vec2( shadowmapTexCoord.x + SHADOWMAP_TEXOFFSET,
									shadowmapTexCoord.y - SHADOWMAP_TEXOFFSET)   ).x;
	rightCol.y = texture2D(tx, vec2( shadowmapTexCoord.x + SHADOWMAP_TEXOFFSET,
									shadowmapTexCoord.y)   ).x;
	rightCol.z = texture2D(tx, vec2( shadowmapTexCoord.x + SHADOWMAP_TEXOFFSET,
									shadowmapTexCoord.y + SHADOWMAP_TEXOFFSET)   ).x;

	// Comparing
	vec3 pixDepth = vec3(lightSpacePixelDepth);
	leftCol  = step( pixDepth, leftCol );	// 0 or 1, depending on comparison
	midCol   = step( pixDepth, midCol );
	rightCol = step( pixDepth, rightCol );

	// Filtering
	vec3 lerpedX1 = mix(leftCol, midCol, fracCoords.x);
	vec3 lerpedX2 = mix(midCol, rightCol, fracCoords.x);

	vec4 interpolateY;
	interpolateY.x = mix(lerpedX1.x, lerpedX1.y, fracCoords.y);
	interpolateY.y = mix(lerpedX1.y, lerpedX1.z, fracCoords.y);
	interpolateY.z = mix(lerpedX2.x, lerpedX2.y, fracCoords.y);
	interpolateY.w = mix(lerpedX2.y, lerpedX2.z, fracCoords.y);

	float sv = dot(interpolateY, vec4(0.25));

#else

	// No filtering
	vec4 mapDepth = texture2D(tx, shadowmapTexCoord.st);
	float sv = (lightSpacePixelDepth >= mapDepth) ? 0.0 : 1.0;

#endif

	sv = mix(shadowColor, 1.0, sv);
	return( vec3(sv) );
}
#endif

// END SUBROUTINES


//------------------------
// FUNCTION_VS Water30VS

uniform mat4  gWorldViewProjMatrix;
uniform mat4  gWorldViewMatrix;
uniform mat4  gWorldViewProjScreenMatrix;
uniform vec4  gModelSpaceCameraPos;
uniform vec4  gSunDir;
uniform vec4  gFakeSunDir;
uniform vec4  gWaterTexMtxParams;
uniform float gFogDensity;
uniform float gFogStart;
uniform float gPlanarFogDensity;
uniform vec4  gPlanarFogEquation;

varying vec2 FoamCoords;
varying vec2 BumpCoords;
varying vec3 ViewVec;
varying vec4 ScreenPos;
varying vec3 TanSunDir;
varying vec3 HalfVector;
varying vec2 FogParams;

#ifdef SHADOWING
uniform mat4 gWorldShadowMatrix;
uniform mat4 gLightProjectedMatrix;
uniform vec4 gShadowXBiasYRange;
varying vec4 ShadowMapCoords0;
varying vec4 ShadowMapCoords1;
varying vec4 ShadowMapCoords2;
#endif

#ifdef SHADOWING_NV
uniform vec4 gShadowPCFOffset0;
uniform vec4 gShadowPCFOffset1;
uniform vec4 gShadowPCFOffset2;
#endif

void main()
{
   ////////////////////////////////////////////////////////////////////////////
   // Transform position
	gl_Position = gl_Vertex * gWorldViewProjMatrix;

   ////////////////////////////////////////////////////////////////////////////
   // Pass diffuse vertex color
   #ifdef NOCOLOR
      gl_FrontColor = vec4(1.0, 1.0, 1.0, 1.0);
   #else
      gl_FrontColor = gl_Color;
   #endif

   ////////////////////////////////////////////////////////////////////////////
   // Generate bump and foam tex coords - multiply tex coords by texture matrix
   // Bump coords
   #ifdef NOCOLOR
      FoamCoords = gl_MultiTexCoord0.st;
	  BumpCoords = FoamCoords;
   #else
      vec2 temp = gl_MultiTexCoord0.st * gWaterTexMtxParams.x;
      BumpCoords.x = temp.x * gWaterTexMtxParams.w + temp.y * -gWaterTexMtxParams.z;
      BumpCoords.y = temp.x * gWaterTexMtxParams.z + temp.y * gWaterTexMtxParams.w + gWaterTexMtxParams.y;
      // Foam coords
      temp = gl_MultiTexCoord0.st * 0.0625;
      FoamCoords.x = temp.x * gWaterTexMtxParams.w + temp.y * -gWaterTexMtxParams.z;
      FoamCoords.y = temp.x * gWaterTexMtxParams.z + temp.y *  gWaterTexMtxParams.w + gWaterTexMtxParams.y;
   #endif

   ////////////////////////////////////////////////////////////////////////////
   // Put view vector in TexCoord1
   vec3 modelSpaceDirToCamera = normalize(gModelSpaceCameraPos.xyz - gl_Vertex.xyz);
   ViewVec = vec3(-modelSpaceDirToCamera.x, modelSpaceDirToCamera.yz);

   ////////////////////////////////////////////////////////////////////////////
   // Get screen space position of vertex
   ScreenPos = gl_Vertex * gWorldViewProjScreenMatrix;

   ////////////////////////////////////////////////////////////////////////////
   // Move sun direction into tangent space.
   vec3 tangent = vec3(1,0,0);
   vec3 binormal = vec3(0,0,1);
   TanSunDir.x = dot(tangent, gSunDir.xyz);
   TanSunDir.y = dot(binormal, gSunDir.xyz);
   TanSunDir.z = dot(gl_Normal, gSunDir.xyz);

   ////////////////////////////////////////////////////////////////////////////
   // Compute H vector.
   vec3 halfAngleVector = normalize(gFakeSunDir.xyz + modelSpaceDirToCamera.xyz);
   
   // Move H vector into tangent space.
   HalfVector.x = dot(tangent.xyz, halfAngleVector.xyz);
   HalfVector.y = dot(binormal.xyz, halfAngleVector.xyz);
   HalfVector.z = dot(gl_Normal.xyz, halfAngleVector.xyz);

   ////////////////////////////////////////////////////////////////////////////
   // Shadow texture coords
#if defined(SHADOWING) && defined(SHADOWING_NV)
	vec4 coords = gl_Vertex * gWorldShadowMatrix;
	coords.y = 1.0 - coords.y;  // Invert y (reading from a GL buffer)
	ShadowMapCoords0 = coords + gShadowPCFOffset0;
	ShadowMapCoords1 = coords + gShadowPCFOffset1;
	ShadowMapCoords2 = coords + gShadowPCFOffset2;
#endif

#if defined(SHADOWING) && !defined(SHADOWING_NV)
	ShadowMapCoords0 = gl_Vertex * gWorldShadowMatrix;

	vec4 lightSpacePosition = gl_Vertex * gLightProjectedMatrix;
	ShadowMapCoords1.x = lightSpacePosition.z * gShadowXBiasYRange.x;
#endif

   ////////////////////////////////////////////////////////////////////////////
   // Fog
	vec4 viewSpacePos = gl_Vertex * gWorldViewMatrix;		// view space (no projection)
	
	FogParams.x = ComputeSquaredExponentialFogDensity( gFogDensity * max(0.0, viewSpacePos.z-gFogStart) / 1000.0);

	float perpDist = dot(viewSpacePos.xyz, gPlanarFogEquation.xyz);
	perpDist -= gPlanarFogEquation.w;

	float t;
	if( perpDist > 0.0 )
		t = 0.0;
	else
	{
		vec3  rayDirection = normalize(-viewSpacePos.xyz);
		float dirAlong     = dot(gPlanarFogEquation.xyz, rayDirection);
		float posAlong     = dot(gPlanarFogEquation.xyz, viewSpacePos.xyz);

		if (abs(dirAlong) < 0.00125)
			dirAlong = 1000.0;
		else
			dirAlong = -1.0 / dirAlong;

		t = gPlanarFogEquation.w - posAlong;
		t *= dirAlong;
	}

	FogParams.y = ComputeSquaredExponentialFogDensity( gPlanarFogDensity * t * 0.01 );   
}


//------------------------
// FUNCTION_PS Water30PS

uniform sampler2D   normalMap;
uniform samplerCube skyCube;
uniform sampler2D   reflection;
uniform sampler2D   foam;
#ifndef NOCOLOR
uniform sampler2D   waveDistortionMap;
#endif

uniform vec4 gWaterBumpScales;
uniform vec4 gWaterColorScales;
uniform vec4 gWaterDeepColor;
uniform vec4 gWaterShallowColor;
uniform vec4 gWaterTexMtxParams;
uniform vec4 gFakeSunParams;
uniform vec4 gPlanarFogColor;
uniform vec4 gFogColor;
#ifdef SCALERGBOUTPUT
uniform float gRGBScale;
#endif

varying vec2 FoamCoords;
varying vec2 BumpCoords;
varying vec3 ViewVec;
varying vec4 ScreenPos;
varying vec3 TanSunDir;
varying vec3 HalfVector;
varying vec2 FogParams;

#ifdef SHADOWING
#ifdef SHADOWING_NV
uniform sampler2DShadow gSampler7;
#endif
#ifndef SHADOWING_NV
uniform sampler2D gSampler7;
#endif
#endif

#ifdef SHADOWING
uniform vec4 gShadowColor;
varying vec4 ShadowMapCoords0;
varying vec4 ShadowMapCoords1;
varying vec4 ShadowMapCoords2;
#endif

void main()
{
   ////////////////////////////////////////////////////////////////////////////
   // Calculate normal
   vec3 NWater = texture2D(normalMap, BumpCoords).rgb;
   vec3 smooth = vec3( 0.5, 0.5, 1.0 );
   NWater = mix( smooth, NWater, gWaterBumpScales.x );
   NWater = normalize( (NWater * 2.0) - 1.0 );

   // Rotate normal map by rotation parameters
   vec3 NWaterWorld;
   NWaterWorld.x = NWater.x * gWaterTexMtxParams.w + NWater.y * gWaterTexMtxParams.z;
   NWaterWorld.y = NWater.x * -gWaterTexMtxParams.z + NWater.y * gWaterTexMtxParams.w;
   NWaterWorld.z = NWater.z;
   
   
   vec3 N;
   vec2 screenPosXY = (ScreenPos.xy / ScreenPos.z);
  
   // MAC_PORT - Because GL rendertarget textures are inverted
   screenPosXY.y = 1.0 - screenPosXY.y;
   
   ////////////////////////////////////////////////////////////////////////////
   // Add wave distortion normal

// HACK!  Turn off wave distortions for NV cards, not working for some reason -- KLC.
#if !defined(NOCOLOR) && !defined(RENDERING_NV)
      // Get wave from distortion map
      vec3 NWaveDistortionWorld = texture2D(waveDistortionMap, screenPosXY).rgb;
      NWaveDistortionWorld = normalize( (NWaveDistortionWorld * 2.0) - 1.0 );
                  
      vec3 normal = NWaveDistortionWorld;
      vec3 tangent = vec3(1.0, 0.0, 0.0);
      vec3 binormal = vec3(0.0, normal.z, -normal.y);			// binormal = cross(normal, tangent);
      tangent = normal * (-normal.x) + vec3(1.0,0.0,0.0);		// tangent = cross(binormal, normal);
    
      N = ((tangent  * NWaterWorld.x + 
            binormal * NWaterWorld.y) / length(normal.yz) + 
            normal   * NWaterWorld.z);
            
#else
      N = NWaterWorld;
#endif
   
   ////////////////////////////////////////////////////////////////////////////
   ////////////////////////////////////////////////////////////////////////////
   
   ////////////////////////////////////////////////////////////////////////////
   // Sky Reflection
   vec3 R = reflect( ViewVec, N );
   vec4 skyReflection = SRGBToLinear(textureCube(skyCube, R));
   
   ////////////////////////////////////////////////////////////////////////////
   // Unit Reflection
   float bumpMax = 0.5 / gWaterBumpScales.x;
   float bumpRatio = clamp(gWaterBumpScales.y, 0.0, bumpMax) * 1.25;
   vec2 factorPerturb = 1.0 - (N.xy * bumpRatio);
   vec2 screenXY = screenPosXY * factorPerturb;
   vec4 localReflection = texture2D(reflection, screenXY);
#ifdef USING_DEPTH_STENCIL
   vec3 totalReflection = mix(skyReflection.rgb, localReflection.rgb, localReflection.a);
#else
   vec3 totalReflection = mix(skyReflection.rgb, localReflection.rgb, 0.75);
#endif

   ////////////////////////////////////////////////////////////////////////////
   // Diffuse
   float nDotL = max(0.0, dot(N, TanSunDir));
   
   ////////////////////////////////////////////////////////////////////////////
   // Specular
   float spec = pow( max(0.0, dot(N, HalfVector)), gFakeSunParams.b) * gFakeSunParams.g;
   
   ////////////////////////////////////////////////////////////////////////////
   // Per-pixel Fresnel calculations
   float blah = 1.0 - max(dot(ViewVec,N.rbg), 0.0);
   float pixelFresnel = clamp( gWaterColorScales.w + (blah * gWaterColorScales.z), 0.0, 1.0 );

   ////////////////////////////////////////////////////////////////////////////
   // Base water color
   vec3 baseColor = gWaterShallowColor.rgb;  // * gSunColor;

   ////////////////////////////////////////////////////////////////////////////
   // Combine it all together
   vec3 color = nDotL * (mix(baseColor, totalReflection, pixelFresnel) + spec);

   ////////////////////////////////////////////////////////////////////////////
   // Foam
   #ifndef NOCOLOR
      vec4 foamColor = SRGBToLinear(texture2D(foam, FoamCoords));
      float temp = clamp( (NWater.y + gWaterBumpScales.z), 0.0, 1.0 );
	  temp = clamp( temp * 4.0, 0.0, 1.0 );
      foamColor.a *= temp;
      color = mix(color, foamColor.rgb, foamColor.a);
   #endif
   color *= gFakeSunParams.r;

/*
   ////////////////////////////////////////////////////////////////////////////
   // Shadow
#if defined(SHADOWING) && defined(SHADOWING_NV)
   factorPerturb = clamp(factorPerturb, 0.99, 1.0);
   vec4 uv0 = ShadowMapCoords0;
   uv0.xy *= factorPerturb;
   vec4 uv1 = ShadowMapCoords1;
   uv1.xy *= factorPerturb;
   vec4 uv2 = ShadowMapCoords2;
   uv2.xy *= factorPerturb;
   color.rgb *= getShadowValueNV(gSampler7, uv0, uv1, uv2, gShadowColor.x) + 0.05;
#endif
*/

#if defined(SHADOWING) && !defined(SHADOWING_NV)
   factorPerturb = clamp(factorPerturb, 0.99, 1.0);
   vec4 uv0 = ShadowMapCoords0;
   uv0.xy *= factorPerturb;
   color.rgb *= getShadowingValue(gSampler7, uv0, ShadowMapCoords1.x, gShadowColor.x) + 0.05;
#endif
   
   ////////////////////////////////////////////////////////////////////////////
   // Vertex Fog
   color.rgb = mix( gPlanarFogColor.rgb, color.rgb, FogParams.y );
   color.rgb = mix( gFogColor.rgb, color.rgb, FogParams.x );

#ifdef SCALERGBOUTPUT
   color *= gRGBScale;
#endif   
   
   ////////////////////////////////////////////////////////////////////////////
   // Return
   gl_FragColor = vec4(color, gl_Color.a);
}
